<?php

class WebSocketServer
{
	public function __construct($host='127.0.0.1', $port=8080)
	{
		$this->host = $host;
		$this->port = $port;

		$this->log('Create server socket ...');
		$this->server = socket_create(AF_INET, SOCK_STREAM, SOL_TCP) or die('socket_create() failed: '.socket_strerror(socket_last_error()));

		$this->log("Bind server to host {$this->host} and port {$this->port} ...");
		socket_bind($this->server, $this->host, $this->port) or die('socket_bind() failed: '.socket_strerror(socket_last_error()));

		$this->log("Listen to incoming websocket connections ...");
		socket_listen($this->server) or die('socket_listen() failed: '.socket_strerror(socket_last_error()));

		$this->sockets = array($this->server);
		$this->rooms = array(
			'/' => array()
		);
		$this->lastMsg = array(
			'socket' => NULL,
			'time' => time(),
			'counter' => 0
		);
	}

	public function start()
	{
		while (true) {
			// create temporary copy of sockets because
			// socket_select() will modify the array

			$this->read = $this->sockets;
			$write = NULL;
			$except = NULL;
			$count = socket_select($this->read, $write, $except, 0, 1000);

			// if it returns false, there's a error
			if ($count === false) {
				echo "socket_select() failed: ".socket_strerror(socket_last_error());
				continue;
			}

			// nothing to do
			if ($count < 1)
				sleep(0.5);

			// new connection or message
			if ($count > 0) {

				// log the time of the message
				$this->lastMsg['time'] = time();

				// new connection
				if (in_array($this->server, $this->read)) {

					// accept socket connection
					$this->sockets[] = $this->lastMsg['socket'] = $sock = socket_accept($this->server);

					// receive request
					$bytes = @socket_recv($sock, $buffer, 2048, 0);

					// find room to join (request path)
					preg_match('/(?<=GET )\S+/', $buffer, $match);

					// add socket to room
					if (!isset($this->rooms[$match[0]])) {

						// create new room
						$this->rooms[$match[0]] = array($sock);
					}else{

						// add socket to room
						$this->rooms[$match[0]][] = $sock;
					}

					$this->log('socket connected ...');
					$this->log('Trying to handshake ...');

					// do handshake to open connection
					$this->do_handshake($buffer, $sock);

					// remove server sock from read-array
					unset($this->read[$this->server]);

					
				}

				//literate over readable sockets
				foreach ($this->read as $sock) {

					// read data from socket and unmask it
					$bytes = @socket_recv($sock, $buffer, 2048, 0);
					$msg = $this->unmask($buffer);

					// log error and continue with next socket
					if ($bytes === false) {
						$this->log('Error at socket_recv(): '.socket_strerror(socket_last_error()));
						continue;
					}

					// remove disconnected socket and continue with next socket
					if ($bytes === 0) {
						$this->log('Disconnected socket removed');
						unset($this->sockets[array_search($sock, $this->sockets)]);
						continue;
					}

					// ignore empty message and continue with next socket
					if ($msg === ''){
						$this->log('Empty message ignored');
						continue;
					}

					// log message
					$this->log('Message received: "'.$msg.'"');

					// on socket close the client send the two bytecodes 3 and 218
					if (strpos($msg, chr(3)) !== false && strpos($msg, chr(218)) !== false) {

						// unset disconnected socket
						unset($this->sockets[array_search($sock, $this->sockets)]);

						// unset disconnected socket from all opened rooms
						foreach ($this->rooms as $key => $r) {
							if (in_array($sock, $r)) {
								unset($this->rooms[$key][$sock]);
							}
						}
						continue;
					}

					// find room in which the sender is
					foreach ($this->rooms as $key => $r) {

						// room found
						if (in_array($sock, $r)) {

							// send the message to all clients in the room except the sender and server socket
							foreach ($r as $socket) {
								if ($socket === $this->server/* || $socket === $sock*/ || $msg === '' ||

									// message is identifier that a socket disconnected
									(strpos($msg, chr(3)) !== false && strpos($msg, chr(218)) !== false))
										continue;

								@socket_write($socket, $this->encode(utf8_encode($this->unmask($buffer))));
							}
						}
					}
				}
			}
		}
	}

	private function do_handshake($buffer, $socket)
	{
		list($resource, $host, $origin, $key) = $this->getheaders($buffer);

		$retkey = base64_encode(sha1($key."258EAFA5-E914-47DA-95CA-C5AB0DC85B11", true));

		$upgrade  = "HTTP/1.1 101 Switching Protocols\r\nUpgrade: websocket\r\nConnection: Upgrade\r\nSec-WebSocket-Accept: {$retkey}\r\n\r\n";
		$this->log("Upgrade:\n".$upgrade);
		@socket_write($socket, $upgrade, strlen($upgrade));

		$this->log("Handshake done; Connection open ...");
	}

	private function getheaders($req)
	{
		$req  = substr($req, 4);
		$res  = substr($req, 0,strpos($req, " HTTP"));
		$req  = substr($req, strpos($req, "Host:")+6);
		$host = substr($req, 0,strpos($req, "\r\n"));
		$req  = substr($req, strpos($req, "Sec-WebSocket-Key: ")+19);
		$key  = trim(substr($req, 0,strpos($req, "\r\n")));
		$req  = substr($req, strpos($req,"Origin:")+8);
		$ori  = substr($req, 0, strpos($req, "\r\n"));

		return array($res, $host, $ori, $key);
	}

	private function encode($text)
	{
		// 0x1 text frame (FIN + opcode)
		$b1 = 0x80 | (0x1 & 0x0f);
		$length = strlen($text);
		
		if($length <= 125)
			$header = pack('CC', $b1, $length);
		elseif($length > 125 && $length < 65536)
			$header = pack('CCS', $b1, 126, $length);
		elseif($length >= 65536)
			$header = pack('CCN', $b1, 127, $length);
	 
		return $header.$text;
	}

	private function unmask($payload) 
	{
		$length = ord($payload[1]) & 127;
	 
		if($length == 126) {
			$masks = substr($payload, 4, 4);
			$data = substr($payload, 8);
			$len = (ord($payload[2]) << 8) + ord($payload[3]);
		}
		elseif($length == 127) {
			$masks = substr($payload, 10, 4);
			$data = substr($payload, 14);
			$len = (ord($payload[2]) << 56) + (ord($payload[3]) << 48) + (ord($payload[4]) << 40) + (ord($payload[5]) << 32) + (ord($payload[6]) << 24) + (ord($payload[7]) << 16) + (ord($payload[8]) << 8) + ord($payload[9]);
		}
		else {
			$masks = substr($payload, 2, 4);
			$data = substr($payload, 6);
			$len = $length;
		}
	 
		$text = '';
		for ($i = 0; $i < $len; ++$i) {
			$text .= $data[$i] ^ $masks[$i%4];
		}
		return $text;
	}

	private function log($text='')
	{
		echo "[server] ".preg_replace("/\n/", "\n\t\t", $text)."\n";
	}
}

?>